<?php
$order_id = $_GET['order_id'] ?? 0;

if (!$order_id) {
    echo "Invalid Order ID!";
    exit;
}

$order = wc_get_order($order_id);

if (!$order) {
    echo "Order not found!";
    exit;
}

$order_total = $order->get_total();
$currency = $order->get_currency();
$order_status = $order->get_status();

if ($order_status == 'completed') {
    wp_redirect($order->get_checkout_order_received_url());
    exit;
}

// Get gateway settings
$gateway_settings = get_option('woocommerce_tender_sdk_payment_settings');
$access_id = $gateway_settings['access_id'] ?? '';
$environment = $gateway_settings['environment'] ?? 'test';

?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Pay via Tender - Order #<?php echo $order->get_order_number(); ?></title>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;600&display=swap" rel="stylesheet">
    <style>
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
            font-family: 'Poppins', sans-serif;
        }

        body {
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 100vh;
            background-color: #f4f7fc;
            padding: 20px;
        }

        .payment-container {
            background: #fff;
            padding: 40px;
            width: 100%;
            max-width: 500px;
            border-radius: 10px;
            box-shadow: 0 4px 10px rgba(0, 0, 0, 0.1);
            text-align: center;
        }

        .logo {
            margin-bottom: 20px;
        }

        h1 {
            font-size: 24px;
            color: #333;
            margin-bottom: 10px;
        }

        .order-info {
            background: #f9f9f9;
            padding: 20px;
            border-radius: 8px;
            margin: 20px 0;
        }

        .order-info p {
            margin: 10px 0;
            font-size: 14px;
            color: #666;
        }

        .order-info .amount {
            font-size: 32px;
            font-weight: 600;
            color: #333;
            margin: 10px 0;
        }

        .loading {
            margin: 20px 0;
        }

        .spinner {
            border: 3px solid #f3f3f3;
            border-top: 3px solid #333;
            border-radius: 50%;
            width: 40px;
            height: 40px;
            animation: spin 1s linear infinite;
            margin: 20px auto;
        }

        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        .cancel-link {
            display: inline-block;
            margin-top: 20px;
            padding: 10px 20px;
            background: #f1f1f1;
            color: #333;
            text-decoration: none;
            border-radius: 5px;
            transition: background 0.3s;
        }

        .cancel-link:hover {
            background: #e0e0e0;
        }

        #tender-sdk-container {
            margin: 20px 0;
        }

        .error-message {
            background: #ffebee;
            color: #c62828;
            padding: 15px;
            border-radius: 5px;
            margin: 20px 0;
        }
    </style>
</head>
<body>
    <div class="payment-container">
        <div class="logo">
            <h1>Tender Payment</h1>
        </div>

        <div class="order-info">
            <p>Order #<?php echo $order->get_order_number(); ?></p>
            <div class="amount"><?php echo wc_price($order_total, ['currency' => $currency]); ?></div>
            <p><?php echo $currency; ?> Payment</p>
        </div>

        <div id="tender-sdk-container">
            <div class="loading">
                <div class="spinner"></div>
                <p>Loading payment gateway...</p>
            </div>
        </div>

        <div id="error-container" style="display: none;">
            <div class="error-message">
                <p id="error-message-text"></p>
            </div>
        </div>

        <a href="<?php echo wc_get_checkout_url(); ?>" class="cancel-link">Cancel Payment</a>
    </div>

    <!-- Import Map to resolve React for ES modules -->
    <script type="importmap">
    {
        "imports": {
            "react": "https://esm.sh/react@18.2.0",
            "react-dom": "https://esm.sh/react-dom@18.2.0",
            "react-dom/client": "https://esm.sh/react-dom@18.2.0/client"
        }
    }
    </script>
    
    <!-- Configuration -->
    <script>
        window.tenderConfig = {
            orderId: <?php echo json_encode($order_id); ?>,
            accessId: <?php echo json_encode($access_id); ?>,
            amount: <?php echo json_encode(floatval($order_total)); ?>,
            currency: <?php echo json_encode($currency); ?>,
            environment: <?php echo json_encode($environment); ?>,
            ajaxUrl: <?php echo json_encode(admin_url('admin-ajax.php')); ?>,
            nonce: <?php echo json_encode(wp_create_nonce('tender_sdk_nonce')); ?>,
            referenceId: <?php echo json_encode('order-' . $order_id . '-' . time()); ?>
        };
    </script>

    <!-- Load Tender SDK and React as ES Modules -->
    <script type="module">
        import React from 'react';
        import ReactDOM from 'react-dom/client';
        import { TenderAgentSdk } from 'https://unpkg.com/@tender-cash/agent-sdk-react@latest/dist/tender-cash-agent-sdk-react.es.js';

        // Handle SDK response
        function handleTenderResponse(response) {
            console.log('Tender SDK Response:', response);

            if (!response || !response.status) {
                console.error('Invalid response from Tender SDK');
                return;
            }

            // Update order status via AJAX
            const formData = new FormData();
            formData.append('action', 'tender_sdk_update_order_status');
            formData.append('security', window.tenderConfig.nonce);
            formData.append('order_id', window.tenderConfig.orderId);
            formData.append('payment_status', response.status);
            formData.append('transaction_id', response.data?.id || '');
            formData.append('amount_paid', response.data?.amountPaid || '');

            fetch(window.tenderConfig.ajaxUrl, {
                method: 'POST',
                body: formData
            })
            .then(res => res.json())
            .then(data => {
                if (data.success && data.data.redirect_url) {
                    window.location.href = data.data.redirect_url;
                } else if (response.status === 'partial-payment') {
                    alert('Partial payment received. Please complete the remaining amount.');
                } else if (response.status === 'error' || response.status === 'cancelled') {
                    document.getElementById('error-container').style.display = 'block';
                    document.getElementById('error-message-text').textContent = 
                        response.message || 'Payment failed. Please try again.';
                }
            })
            .catch(error => {
                console.error('Error updating order:', error);
                document.getElementById('error-container').style.display = 'block';
                document.getElementById('error-message-text').textContent = 
                    'Error processing payment. Please contact support.';
            });
        }

        // Initialize Tender SDK
        try {
            // Hide loading spinner
            document.querySelector('.loading').style.display = 'none';

            // Create React element with SDK
            const container = document.getElementById('tender-sdk-container');
            const root = ReactDOM.createRoot(container);
            
            const sdkElement = React.createElement(TenderAgentSdk, {
                accessId: window.tenderConfig.accessId,
                fiatCurrency: window.tenderConfig.currency,
                env: window.tenderConfig.environment,
                amount: window.tenderConfig.amount,
                referenceId: window.tenderConfig.referenceId,
                paymentExpirySeconds: 1800, // 30 minutes
                theme: 'light',
                onEventResponse: handleTenderResponse
            });

            root.render(sdkElement);
        } catch (error) {
            console.error('Error initializing Tender SDK:', error);
            document.querySelector('.loading').style.display = 'none';
            document.getElementById('error-container').style.display = 'block';
            document.getElementById('error-message-text').textContent = 
                'Failed to load payment gateway. Please refresh the page or contact support.';
        }
    </script>
</body>
</html>
