<?php

class WC_Tender_SDK_Gateway extends WC_Payment_Gateway {

    private $access_id;
    private $access_secret;
    private $agent_id;
    private $environment;
    private $api_base_url;

    public function __construct() {
        $this->id = 'tender_sdk_payment';
        $this->method_title = __('Tender SDK Payment','woocommerce-tender-sdk-gateway');
        $this->title = __('Pay with Crypto','woocommerce-tender-sdk-gateway');
        $this->has_fields = false;
        $this->init_form_fields();
        $this->init_settings();
        $this->enabled = $this->get_option('enabled');
        $this->title = $this->get_option('title');
        $this->description = $this->get_option('description');
        $this->access_id = $this->get_option('access_id');
        $this->access_secret = $this->get_option('access_secret');
        $this->agent_id = $this->get_option('agent_id');
        $this->environment = $this->get_option('environment', 'test');
        
        // Set API base URL based on environment
        $this->api_base_url = $this->environment === 'live' 
            ? 'https://api.tender.cash' 
            : 'https://stagapi.tender.cash';

        // Save admin options
        add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
    }

    public function init_form_fields() {
        $this->form_fields = array(
            'enabled' => array(
                'title' => __( 'Enable/Disable', 'woocommerce-tender-sdk-gateway' ),
                'type' => 'checkbox',
                'label' => __( 'Enable Tender SDK Payment', 'woocommerce-tender-sdk-gateway' ),
                'default' => 'yes'
            ),
            'title' => array(
                'title' => __( 'Payment Title', 'woocommerce-tender-sdk-gateway' ),
                'type' => 'text',
                'description' => __( 'Payment Method Name', 'woocommerce-tender-sdk-gateway' ),
                'default' => __( 'Pay with Crypto', 'woocommerce-tender-sdk-gateway' ),
                'desc_tip' => true,
            ),
            'description' => array(
                'title' => __( 'Payment Note', 'woocommerce-tender-sdk-gateway' ),
                'type' => 'textarea',
                'css' => 'width:500px;',
                'default' => 'Pay securely with cryptocurrency using Tender',
                'description' => __( 'The message which you want it to appear to the customer in the checkout page.', 'woocommerce-tender-sdk-gateway' ),
            ),
            'access_id' => array(
                'title' => __( 'Access ID', 'woocommerce-tender-sdk-gateway' ),
                'type' => 'text',
                'description' => __( 'Your Tender Access ID from the merchant dashboard', 'woocommerce-tender-sdk-gateway' ),
                'default' => '',
                'desc_tip' => true,
            ),
            'access_secret' => array(
                'title' => __( 'Access Secret', 'woocommerce-tender-sdk-gateway' ),
                'type' => 'password',
                'description' => __( 'Your Tender Access Secret from the merchant dashboard', 'woocommerce-tender-sdk-gateway' ),
                'default' => '',
                'desc_tip' => true,
            ),
            'agent_id' => array(
                'title' => __( 'Agent ID', 'woocommerce-tender-sdk-gateway' ),
                'type' => 'text',
                'description' => __( 'Your Tender Agent ID from the merchant dashboard', 'woocommerce-tender-sdk-gateway' ),
                'default' => '',
                'desc_tip' => true,
            ),
            'environment' => array(
                'title' => __( 'Environment', 'woocommerce-tender-sdk-gateway' ),
                'type' => 'select',
                'description' => __( 'Select the environment (test or live)', 'woocommerce-tender-sdk-gateway' ),
                'default' => 'test',
                'options' => array(
                    'test' => __( 'Test/Staging', 'woocommerce-tender-sdk-gateway' ),
                    'live' => __( 'Live/Production', 'woocommerce-tender-sdk-gateway' )
                ),
                'desc_tip' => true,
            ),
        );
    }

    public function process_payment($order_id) {
        $order = wc_get_order($order_id);
        
        // Mark order as pending
        $order->update_status('pending', __('Awaiting Tender payment', 'woocommerce-tender-sdk-gateway'));
        
        // Reduce stock levels
        wc_reduce_stock_levels($order_id);
        
        // Remove cart
        WC()->cart->empty_cart();
        
        return [
            'result' => 'success',
            'redirect' => add_query_arg('order_id', $order_id, home_url('/tender-sdk-payment/'))
        ];
    }
}

// Register custom rewrite rules for payment page
add_action('init', function () {
    add_rewrite_rule('tender-sdk-payment/?$', 'index.php?tender_sdk_payment_page=1', 'top');
});

add_filter('query_vars', function ($vars) {
    $vars[] = 'tender_sdk_payment_page';
    return $vars;
});

add_action('template_redirect', function () {
    if (get_query_var('tender_sdk_payment_page') == 1) {
        include plugin_dir_path(__FILE__) . 'templates/tender-sdk-payment.php';
        exit;
    }
});

function tender_sdk_plugin_activate() {
    flush_rewrite_rules();
}
register_activation_hook(__FILE__, 'tender_sdk_plugin_activate');

// AJAX handler to update order status after payment
add_action('wp_ajax_tender_sdk_update_order_status', 'tender_sdk_update_order_status');
add_action('wp_ajax_nopriv_tender_sdk_update_order_status', 'tender_sdk_update_order_status');

function tender_sdk_update_order_status() {
    check_ajax_referer('tender_sdk_nonce', 'security');
    
    $order_id = intval($_POST['order_id']);
    $payment_status = sanitize_text_field($_POST['payment_status']);
    $transaction_id = sanitize_text_field($_POST['transaction_id']);
    $amount_paid = sanitize_text_field($_POST['amount_paid']);
    
    if (!$order_id) {
        wp_send_json_error(['message' => 'Invalid Order ID']);
    }
    
    $order = wc_get_order($order_id);
    
    if (!$order) {
        wp_send_json_error(['message' => 'Order not found']);
    }
    
    // Store transaction details
    $order->update_meta_data('tender_transaction_id', $transaction_id);
    $order->update_meta_data('tender_amount_paid', $amount_paid);
    $order->update_meta_data('tender_payment_status', $payment_status);
    
    switch ($payment_status) {
        case 'completed':
            $order->payment_complete($transaction_id);
            $order->add_order_note(__('Payment completed via Tender SDK.', 'woocommerce-tender-sdk-gateway'));
            break;
            
        case 'overpayment':
            $order->payment_complete($transaction_id);
            $order->add_order_note(sprintf(__('Overpayment received via Tender SDK. Amount paid: %s', 'woocommerce-tender-sdk-gateway'), $amount_paid));
            break;
            
        case 'partial-payment':
            $order->update_status('on-hold');
            $order->add_order_note(sprintf(__('Partial payment received via Tender SDK. Amount paid: %s', 'woocommerce-tender-sdk-gateway'), $amount_paid));
            break;
            
        default:
            $order->update_status('pending');
            $order->add_order_note(__('Payment pending via Tender SDK.', 'woocommerce-tender-sdk-gateway'));
    }
    
    $order->save();
    
    wp_send_json_success([
        'status' => $payment_status,
        'redirect_url' => $order->get_checkout_order_received_url()
    ]);
}
